import { auth } from '@/app/auth';
import { redirect } from 'next/navigation';
import LearningPage from '@/app/components/LearningPage';
import { getUser } from '@/app/db';
import { getNextIndexForUser, listWordsByBook } from '@/app/db';

interface PageProps {
  params: {
    bookId: string;
  };
}

export default async function LearnPage({ params }: PageProps) {
  const session = await auth();
  if (!session) redirect('/me?auth=login');

  // Resolve user id (from session or by email lookup)
  let userId: number | null = null;
  const email = (session!.user as any)?.email as string | undefined;
  if ((session!.user as any)?.id) {
    userId = Number((session!.user as any).id);
  } else if (email) {
    const rows = await getUser(email);
    if (rows && rows.length > 0) userId = Number(rows[0].id);
  }
  if (!userId) redirect('/me?auth=login');

  // Compute start index based on user_book_progress
  const nextIdx = await getNextIndexForUser(userId, params.bookId);
  // Fetch words for this book
  const dbWords = await listWordsByBook(params.bookId);

  // Normalize for client component
  const words = dbWords.map((w) => ({
    id: w.id,
    idx: 0, // will be determined by array index
    wordRank: w.wordRank ?? 0,
    headWord: w.headWord ?? '',
    content: w.content,
    bookId: w.bookId ?? params.bookId,
  }));

  return (
    <LearningPage
      bookId={params.bookId}
      session={session!}
      initialIdx={Math.min(nextIdx, Math.max(words.length - 1, 0))}
      serverWords={words}
    />
  );
}
